print('######################################################################################################')
print('############## You are using Modification and Annotation in Proteins (MAP) Script ####################')
print('##################### MAP Program, developed by Prof G. P. S. Raghava group. #########################')
print('############ Please cite: MAP; available at https://webs.iiitd.edu.in/raghava/maprepo/  ##############')
print('######################################################################################################')
import streamlit as st
import pandas as pd
import re

# --- Page Configuration ---
st.set_page_config(page_title="MAP-to-FASTA Tool", layout="centered")
st.title("🧬 MAP-to-FASTA Converter")
st.markdown("""
This tool cleans sequences with `{}`-style annotations and exports them in **CSV** or **FASTA** format.
""")

# --- Style ---
st.markdown(
    """
<style>
.download-button {
    background-color: #4CAF50; /* Green */
    border: none;
    color: white;
    padding: 10px 24px;
    text-align: center;
    text-decoration: none;
    display: inline-block;
    font-size: 16px;
    margin: 4px 2px;
    cursor: pointer;
    border-radius: 8px;
    transition: background-color 0.3s ease;
}

.download-button:hover {
    background-color: #367c39; /* Darker green */
}

.error-message {
    color: #FF0000;
    font-size: 16px;
    font-weight: bold;
    margin-top: 10px;
}
</style>
""",
    unsafe_allow_html=True,
)

# --- Upload CSV ---
uploaded_file = st.file_uploader("📂 Upload your MAP/Tagged CSV file", type=["csv"])

# --- Metadata Inputs ---
org = st.text_input("Organism", "undefine")
func = st.text_input("Function", "unknown")
prefix = st.text_input("FASTA Header Prefix", "Sample")

# --- Output Format ---
output_format = st.radio("Select Output Format", ["CSV", "FASTA"], horizontal=True)

# --- Conversion Logic ---
if uploaded_file and st.button("🚀 Convert"):
    try:
        # Load file
        df = pd.read_csv(uploaded_file, header=None)

        # Detect header or assign default
        if df.iloc[0].astype(str).str.contains(r'\{.*?\}').any():
            df.columns = ['MAP_seq']
        else:
            df.columns = df.iloc[0]
            df = df[1:].reset_index(drop=True)

        # Detect column with `{}` tags
        candidate_col = None
        for col in df.columns:
            if df[col].astype(str).str.contains(r'\{.*?\}').any():
                candidate_col = col
                break

        if not candidate_col:
            st.error("❌ No column with `{}` tags found.")
        else:
            # Clean sequences
            df['Clean Sequence'] = df[candidate_col].apply(lambda x: re.sub(r'\{.*?\}', '', str(x)))

            # CSV output
            if output_format == "CSV":
                csv = df.to_csv(index=False).encode('utf-8')
                st.markdown(
                    f'<a href="data:text/csv;charset=utf-8,{csv.decode()}" download="{prefix}_cleaned.csv" class="download-button">💾 Download Cleaned CSV</a>',
                    unsafe_allow_html=True,
                )

            # FASTA output
            else:
                fasta = "\n".join([
                    f">{prefix}_{i+1} {{org:{org}}} {{func:{func}}}\n{seq}"
                    for i, seq in enumerate(df['Clean Sequence'])
                ])
                st.markdown(
                    f'<a href="data:text/plain;charset=utf-8,{fasta}" download="{prefix}.fasta" class="download-button">💾 Download FASTA</a>',
                    unsafe_allow_html=True,
                )

    except Exception as e:
        st.error(f"<div class='error-message'>❌ An error occurred: {e}</div>", unsafe_allow_html=True)
