import streamlit as st
import pandas as pd
import re
from utils import get_smi_from_map


def main():
    st.title("MAP to SMILES Format Converter")

    # Tabs for different functionalities
    tab1, = st.tabs(["MAP to SMILES"])

    with tab1:
        st.header("Convert MAP to SMILES")

        # Example MAP sequences
        map_examples = "\n".join([
            "{nnr:Abu}{nnr:Sar}L{nnm:N-Methyl}VL{nnm:N-Methyl}AA{d}L{nnm:N-Methyl}L{nnm:N-Methyl}V{nnm:N-Methyl}{nnr:Mebmt}{cyc:N-C}",
            "L{d}L{d}LL{d}PY{cyc:N-C}",
            "L{d}L{d}L{d}L{d}PY{cyc:N-C}",
            "LLLL{d}PY{cyc:N-C}",
            "LL{d}L{d}L{d}PY{cyc:N-C}"
        ])

        if st.button("Example sequence", key="smiles_example_button"):
            st.session_state.map_input_smiles = map_examples

        # Text input
        map_input_smiles = st.text_area(
            "Enter MAP sequences (one per line):",
            value=st.session_state.get("map_input_smiles", ""),
            key="map_smiles_input_text"
        )

        # File uploader
        uploaded_file = st.file_uploader("Or upload a .txt file with one MAP sequence per line:", type=["txt"])

        if st.button("Convert", key="map_to_smiles_convert"):
            map_lines = []

            # From text area
            if map_input_smiles:
                map_lines.extend([line.strip() for line in map_input_smiles.strip().split('\n') if line.strip()])

            # From file
            if uploaded_file is not None:
                file_lines = uploaded_file.read().decode("utf-8").splitlines()
                map_lines.extend([line.strip() for line in file_lines if line.strip()])

            if map_lines:
                smiles_outputs = [get_smi_from_map(line) for line in map_lines]
                smiles_output = "\n".join(smiles_outputs)

                st.success("Conversion Successful!")
                st.text_area("SMILES Format:", value=smiles_output, height=150, key="map_smiles_output")

                st.download_button(
                    label="Download SMILES Format",
                    data=smiles_output,
                    file_name='output_smiles.txt',
                    mime='text/plain',
                    key="download_smiles"
                )
            else:
                st.error("Please provide MAP input via text area or file.")

if __name__ == "__main__":
    main()