import streamlit as st
import pandas as pd
import re
from utils import process_HELM_seq, convert_map_to_helm_sequence
st.set_page_config(layout="wide")
def main():
    st.title("MAP to HELM Format Converter")

    # Tabs for different functionalities
    tab1, = st.tabs(["MAP to HELM"])

    with tab1:
        st.header("Convert MAP to HELM")

        # Example sequences
        map_examples = "\n".join([
            "{nnr:Abu}{nnr:Sar}L{nnm:N-Methyl}VL{nnm:N-Methyl}AA{d}L{nnm:N-Methyl}L{nnm:N-Methyl}V{nnm:N-Methyl}{nnr:Mebmt}{cyc:N-C}",
            "L{d}L{d}LL{d}PY{cyc:N-C}",
            "L{d}L{d}L{d}L{d}PY{cyc:N-C}",
            "LLLL{d}PY{cyc:N-C}",
            "LL{d}L{d}L{d}PY{cyc:N-C}"
        ])
        id_examples = "\n".join(["1", "2", "3", "4", "5"])

        if st.button("Show Examples", key="map_example_button"):
            st.session_state.map_input = map_examples
            st.session_state.id_input = id_examples

        # Manual input
        col1, col2 = st.columns([7, 3])  # 70% and 30% width

        with col1:
            map_input = st.text_area("Enter MAP sequences (one per line):", value=st.session_state.get("map_input", ""), key="map_input_text")

        with col2:
            id_input = st.text_area("Enter a base ID for the peptide(s):", value=st.session_state.get("id_input", ""), key="map_id_input_text")

        # File input
        uploaded_file = st.file_uploader("Or upload a .txt file with each line as: MAP_SEQUENCE,ID", type=["txt"])

        if st.button("Convert", key="map_to_helm_convert"):
            map_lines = []
            id_lines = []

            # Collect from manual input
            if map_input and id_input:
                map_lines.extend([line.strip() for line in map_input.strip().split('\n') if line.strip()])
                id_lines.extend([line.strip() for line in id_input.strip().split('\n') if line.strip()])

            # Collect from uploaded file
            if uploaded_file is not None:
                file_lines = uploaded_file.read().decode("utf-8").splitlines()
                for line in file_lines:
                    if ',' in line:
                        map_seq, map_id = map(str.strip, line.split(",", 1))
                        if map_seq and map_id:
                            map_lines.append(map_seq)
                            id_lines.append(map_id)

            # Validate input
            if map_lines and id_lines and len(map_lines) == len(id_lines):
                helm_outputs = []
                for line, i in zip(map_lines, id_lines):
                    helm_seq = convert_map_to_helm_sequence(line, i)
                    helm_outputs.append(process_HELM_seq(helm_seq, i))
                helm_output = "\n".join(helm_outputs)

                st.success("Conversion Successful!")
                st.text_area("HELM Format:", value=helm_output, height=150, key="map_helm_output")
                st.download_button(
                    label="Download HELM Format",
                    data=helm_output,
                    file_name='output_helm.txt',
                    mime='text/plain',
                    key="download_helm"
                )
            else:
                st.error("Please provide matching MAP and ID inputs either manually or via a file.")

if __name__ == "__main__":
    main()