import argparse
import os
import pandas as pd
import re
from utils import process_HELM_seq, convert_map_to_helm_sequence 


def main():
    parser = argparse.ArgumentParser(description="HELM-MAP-SMILES Format Converter")
    parser.add_argument("mode", choices=[ "map_to_helm"], help="Conversion mode")
    parser.add_argument("--input", required=True, help="Input string or input file path")
    parser.add_argument("--output", help="Output file path (required if input is a file)")
    parser.add_argument("--id", help="Peptide ID(s) for MAP to HELM (comma-separated for batch)")

    args = parser.parse_args()
    is_file = os.path.isfile(args.input)

    # Handle MAP to HELM
    if args.mode == 'map_to_helm':
      if is_file:
          if not args.output:
              print("Error: Please specify --output for saving the converted HELM format.")
          else:
              with open(args.input, 'r') as infile, open(args.output, 'w') as outfile:
                  for line in infile:
                      line = line.strip()
                      if not line or ',' not in line:
                          continue
                      try:
                          map_seq, peptide_id = map(str.strip, line.split(',', 1))
                          helm_seq = convert_map_to_helm_sequence(map_seq, peptide_id)
                          result = process_HELM_seq(helm_seq, peptide_id)
                          outfile.write(result + '\n')
                      except Exception as e:
                          print(f"Skipping invalid line: {line} | Error: {e}")
              print(f"Conversion complete. Output saved to {args.output}")
      else:
          if not args.id:
              print("Error: --id is required for single sequence MAP to HELM conversion.")
          else:
              helm_seq = convert_map_to_helm_sequence(args.input, args.id)
              result = process_HELM_seq(helm_seq, args.id)
              print(result)


if __name__ == "__main__":
    main()