import streamlit as st
import pandas as pd
import re
from utils import helm_to_map

def main():
    st.title("HELM to MAP Format Converter")
    tab1, = st.tabs(["HELM to MAP"])

    with tab1:
        st.header("Convert HELM to MAP")

        # Example sequences
        helm_examples = "\n".join([
            "PEPTIDE1{[Abu].[Sar].[meL].V.[meL].A.[dA].[meL].[meL].[meV].[Me_Bmt(E)]}$PEPTIDE1,PEPTIDE1,1:R1-11:R2$$$",
            "PEPTIDE2{[dL].[dL].L.[dL].P.Y}$PEPTIDE2,PEPTIDE2,1:R1-6:R2$$$",
            "PEPTIDE3{[dL].[dL].[dL].[dL].P.Y}$PEPTIDE3,PEPTIDE3,1:R1-6:R2$$$",
            "PEPTIDE4{L.L.L.[dL].P.Y}$PEPTIDE4,PEPTIDE4,1:R1-6:R2$$$",
            "PEPTIDE5{L.[dL].[dL].[dL].P.Y}$PEPTIDE5,PEPTIDE5,1:R1-6:R2$$$"
        ])

        if st.button("Example sequence", key="helm_example_button"):
            st.session_state.helm_input = helm_examples

        # Text input
        helm_input = st.text_area("Enter HELM notations (one per line):", value=st.session_state.get("helm_input", ""), key="helm_input_text")

        # File upload
        uploaded_file = st.file_uploader("Or upload a .txt file with one HELM sequence per line:", type=["txt"])

        if st.button("Convert", key="helm_to_map_convert"):
            # Collect HELM lines from text area
            input_lines = [line.strip() for line in helm_input.strip().split('\n') if line.strip()] if helm_input else []


            if uploaded_file is not None:
                file_lines = uploaded_file.read().decode("utf-8").splitlines()
                file_lines = [line.strip() for line in file_lines if line.strip()]
                input_lines.extend(file_lines)

            if input_lines:
                map_outputs = [helm_to_map(line) for line in input_lines]
                map_output = "\n".join(map_outputs)

                st.success("Conversion Successful!")
                st.text_area("MAP Format:", value=map_output, height=150, key="helm_map_output")

                st.download_button(
                    label="Download MAP Format",
                    data=map_output,
                    file_name='output_map.txt',
                    mime='text/plain',
                    key="download_map"
                )
            else:
                st.error("Please enter a HELM sequence or upload a valid file.")

if __name__ == "__main__":
    main()